from axelrod.action import Action, actions_to_str

from axelrod.player import Player

from axelrod.strategy_transformers import (
    FinalTransformer,
    TrackHistoryTransformer,
)

C, D = Action.C, Action.D

class NTitsForMTats(Player):
    """
    A parameterizable Tit-for-Tat,
    The arguments are:
    1) M: the number of defection before retaliation
    2) N: the number of retaliations

    Names:

    - N Tit(s) For M Tat(s): Original name by Marc Harper
    """

    name = "N Tit(s) For M Tat(s)"
    classifier = {
        "memory_depth": float("inf"),
        "stochastic": False,
        "long_run_time": False,
        "inspects_source": False,
        "manipulates_source": False,
        "manipulates_state": False,
    }

    def __init__(self, N: int = 3, M: int = 2) -> None:
        """
        Parameters
        ----------
        N: int
            Number of retaliations
        M: int
            Number of defection before retaliation

        Special Cases
        -------------
        NTitsForMTats(1,1) is equivalent to TitForTat
        NTitsForMTats(1,2) is equivalent to TitFor2Tats
        NTitsForMTats(2,1) is equivalent to TwoTitsForTat
        NTitsForMTats(0,*) is equivalent to Cooperator
        NTitsForMTats(*,0) is equivalent to Defector
        """
        super().__init__()
        self.N = N
        self.M = M
        self.classifier["memory_depth"] = max([M, N])
        self.retaliate_count = 0

    def strategy(self, opponent: Player) -> Action:
        """Actual strategy definition that determines player's action."""
        # if opponent defected consecutively M times, start the retaliation
        if not self.M or opponent.history[-self.M :].count(D) == self.M:
            self.retaliate_count = self.N
        if self.retaliate_count:
            self.retaliate_count -= 1
            return D
        return C